/*
 * Decompiled with CFR 0.152.
 */
package dan200.computercraft.core.computer.computerthread;

import com.google.common.annotations.VisibleForTesting;
import com.google.errorprone.annotations.Keep;
import com.google.errorprone.annotations.concurrent.GuardedBy;
import dan200.computercraft.core.Logging;
import dan200.computercraft.core.computer.TimeoutState;
import dan200.computercraft.core.computer.computerthread.ComputerScheduler;
import dan200.computercraft.core.computer.computerthread.ManagedTimeoutState;
import dan200.computercraft.core.metrics.Metrics;
import dan200.computercraft.core.metrics.MetricsObserver;
import dan200.computercraft.core.util.ThreadUtils;
import java.util.Objects;
import java.util.TreeSet;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.atomic.AtomicReferenceFieldUpdater;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.LockSupport;
import java.util.concurrent.locks.ReentrantLock;
import org.jspecify.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ComputerThread
implements ComputerScheduler {
    private static final Logger LOG = LoggerFactory.getLogger(ComputerThread.class);
    private static final ThreadFactory monitorFactory = ThreadUtils.builder("Computer-Monitor").setPriority(7).build();
    private static final ThreadFactory workerFactory = ThreadUtils.lowPriorityFactory("Computer-Worker");
    private static final long MONITOR_WAKEUP = TimeUnit.MILLISECONDS.toNanos(100L);
    private static final long DEFAULT_LATENCY = TimeUnit.MILLISECONDS.toNanos(50L);
    private static final long DEFAULT_MIN_PERIOD = TimeUnit.MILLISECONDS.toNanos(5L);
    private static final long LATENCY_MAX_TASKS = DEFAULT_LATENCY / DEFAULT_MIN_PERIOD;
    private static final long REPORT_DEBOUNCE = TimeUnit.SECONDS.toNanos(1L);
    private final ReentrantLock threadLock = new ReentrantLock();
    private static final int RUNNING = 0;
    private static final int STOPPING = 1;
    private static final int CLOSED = 2;
    private final AtomicInteger state = new AtomicInteger(0);
    private @Nullable Thread monitor;
    @GuardedBy(value="threadLock")
    private final WorkerThread[] workers;
    @GuardedBy(value="threadLock")
    private int workerCount = 0;
    private final Condition shutdown = this.threadLock.newCondition();
    private final long latency;
    private final long minPeriod;
    private final ReentrantLock computerLock = new ReentrantLock();
    @GuardedBy(value="computerLock")
    private final Condition workerWakeup = this.computerLock.newCondition();
    @GuardedBy(value="computerLock")
    private final Condition monitorWakeup = this.computerLock.newCondition();
    private final AtomicInteger idleWorkers = new AtomicInteger(0);
    @GuardedBy(value="computerLock")
    private final TreeSet<ExecutorImpl> computerQueue = new TreeSet(ComputerThread::compareExecutors);
    private long minimumVirtualRuntime = 0L;

    private static int compareExecutors(ExecutorImpl a, ExecutorImpl b) {
        if (a == b) {
            return 0;
        }
        long at = a.virtualRuntime;
        long bt = b.virtualRuntime;
        if (at == bt) {
            return Integer.compare(a.hashCode(), b.hashCode());
        }
        return at < bt ? -1 : 1;
    }

    public ComputerThread(int threadCount) {
        this.workers = new WorkerThread[threadCount];
        int factor = 64 - Long.numberOfLeadingZeros(this.workers.length);
        this.latency = DEFAULT_LATENCY * (long)factor;
        this.minPeriod = DEFAULT_MIN_PERIOD * (long)factor;
    }

    @Override
    public ComputerScheduler.Executor createExecutor(ComputerScheduler.Worker worker, MetricsObserver metrics) {
        return new ExecutorImpl(worker, metrics);
    }

    @GuardedBy(value="threadLock")
    private void addWorker(int index) {
        LOG.trace("Spawning new worker {}.", (Object)index);
        this.workers[index] = new WorkerThread(index);
        this.workers[index].owner.start();
        ++this.workerCount;
    }

    private int workerCount() {
        return this.workerCount;
    }

    private WorkerThread[] workersReadOnly() {
        return this.workers;
    }

    @GuardedBy(value="computerLock")
    private void ensureRunning() {
        block6: {
            if (this.monitor != null && (this.idleWorkers.get() > 0 || this.workerCount() == this.workersReadOnly().length)) {
                return;
            }
            this.threadLock.lock();
            try {
                LOG.trace("Possibly spawning a worker or monitor.");
                if (this.monitor == null || !this.monitor.isAlive()) {
                    this.monitor = monitorFactory.newThread(new Monitor());
                    this.monitor.start();
                }
                if (this.idleWorkers.get() != 0 && this.workerCount >= this.workers.length) break block6;
                for (int i = 0; i < this.workers.length; ++i) {
                    if (this.workers[i] != null) continue;
                    this.addWorker(i);
                    break;
                }
            }
            finally {
                this.threadLock.unlock();
            }
        }
    }

    private void advanceState(int newState) {
        int current;
        while ((current = this.state.get()) < newState && !this.state.compareAndSet(current, newState)) {
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public boolean stop(long timeout, TimeUnit unit) throws InterruptedException {
        this.advanceState(1);
        this.threadLock.lock();
        try {
            for (WorkerThread worker : this.workers) {
                ExecutorImpl executor;
                if (worker == null || (executor = worker.currentExecutor.get()) == null) continue;
                executor.timeout.hardAbort();
            }
        }
        finally {
            this.threadLock.unlock();
        }
        this.computerLock.lock();
        try {
            this.workerWakeup.signalAll();
        }
        finally {
            this.computerLock.unlock();
        }
        long timeoutNs = unit.toNanos(timeout);
        this.threadLock.lock();
        try {
            while (this.workerCount > 0) {
                if (timeoutNs <= 0L) {
                    int n = 0;
                    return n != 0;
                }
                timeoutNs = this.shutdown.awaitNanos(timeoutNs);
            }
        }
        finally {
            this.threadLock.unlock();
        }
        this.advanceState(2);
        this.computerLock.lock();
        try {
            this.monitorWakeup.signal();
        }
        finally {
            this.computerLock.unlock();
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void queue(ExecutorImpl executor) {
        this.computerLock.lock();
        try {
            if (this.state.get() != 0) {
                throw new IllegalStateException("ComputerThread is no longer running");
            }
            this.ensureRunning();
            this.updateRuntimes(null);
            long newRuntime = this.minimumVirtualRuntime;
            newRuntime = executor.virtualRuntime == 0L ? (newRuntime += this.scaledPeriod()) : (newRuntime -= this.latency / 2L);
            executor.virtualRuntime = Math.max(newRuntime, executor.virtualRuntime);
            boolean wasBusy = this.isBusy();
            this.computerQueue.add(executor);
            this.workerWakeup.signal();
            if (!wasBusy && this.isBusy()) {
                this.monitorWakeup.signal();
            }
        }
        finally {
            this.computerLock.unlock();
        }
    }

    @GuardedBy(value="computerLock")
    private void updateRuntimes(@Nullable ExecutorImpl current) {
        long minRuntime = Long.MAX_VALUE;
        if (!this.computerQueue.isEmpty()) {
            minRuntime = this.computerQueue.first().virtualRuntime;
        }
        long now = System.nanoTime();
        int tasks = 1 + this.computerQueue.size();
        for (WorkerThread runner : this.workersReadOnly()) {
            ExecutorImpl executor;
            if (runner == null || (executor = runner.currentExecutor.get()) == null) continue;
            minRuntime = Math.min(minRuntime, executor.virtualRuntime += (now - executor.vRuntimeStart) / (long)tasks);
            executor.vRuntimeStart = now;
        }
        if (current != null) {
            minRuntime = Math.min(minRuntime, current.virtualRuntime += (now - current.vRuntimeStart) / (long)tasks);
        }
        if (minRuntime > this.minimumVirtualRuntime && minRuntime < Long.MAX_VALUE) {
            this.minimumVirtualRuntime = minRuntime;
        }
    }

    private void afterWork(ExecutorImpl executor) {
        this.computerLock.lock();
        try {
            this.updateRuntimes(executor);
            if (!executor.afterWork() || this.state.get() != 0) {
                return;
            }
            this.computerQueue.add(executor);
            this.workerWakeup.signal();
        }
        finally {
            this.computerLock.unlock();
        }
    }

    private int computerQueueSize() {
        return this.computerQueue.size();
    }

    long scaledPeriod() {
        int count = 1 + this.computerQueueSize();
        return (long)count < LATENCY_MAX_TASKS ? this.latency / (long)count : this.minPeriod;
    }

    @VisibleForTesting
    boolean hasPendingWork() {
        return this.computerQueueSize() > 0;
    }

    private boolean isBusy() {
        return this.computerQueueSize() > this.idleWorkers.get();
    }

    @VisibleForTesting
    boolean isFullyIdle() {
        return this.computerQueueSize() == 0 && this.idleWorkers.get() >= this.workerCount();
    }

    private void workerFinished(WorkerThread worker) {
        if (!worker.running.getAndSet(false)) {
            return;
        }
        LOG.trace("Worker {} finished.", (Object)worker.index);
        ExecutorImpl executor = worker.currentExecutor.getAndSet(null);
        if (executor != null) {
            executor.afterWork();
        }
        this.threadLock.lock();
        try {
            --this.workerCount;
            if (this.workers[worker.index] != worker) {
                assert (false) : "workerFinished but inconsistent worker";
                LOG.error("Worker {} closed, but new runner has been spawned.", (Object)worker.index);
            } else if (this.state.get() == 0 || this.state.get() == 1 && this.hasPendingWork()) {
                this.addWorker(worker.index);
                ++this.workerCount;
            } else {
                this.workers[worker.index] = null;
            }
        }
        finally {
            this.threadLock.unlock();
        }
    }

    private final class ExecutorImpl
    implements ComputerScheduler.Executor {
        public static final AtomicReferenceFieldUpdater<ExecutorImpl, ExecutorState> STATE = AtomicReferenceFieldUpdater.newUpdater(ExecutorImpl.class, ExecutorState.class, "$state");
        final ComputerScheduler.Worker worker;
        private final MetricsObserver metrics;
        final TimeoutImpl timeout;
        @Keep
        private volatile ExecutorState $state = ExecutorState.IDLE;
        long virtualRuntime = 0L;
        long vRuntimeStart;

        ExecutorImpl(ComputerScheduler.Worker worker, MetricsObserver metrics) {
            this.worker = worker;
            this.metrics = metrics;
            this.timeout = new TimeoutImpl();
        }

        void beforeWork() {
            this.vRuntimeStart = System.nanoTime();
            this.timeout.startTimer(ComputerThread.this.scaledPeriod());
        }

        boolean afterWork() {
            this.timeout.reset();
            this.metrics.observe(Metrics.COMPUTER_TASKS, this.timeout.getExecutionTime());
            ExecutorState state = STATE.getAndUpdate(this, ExecutorState::requeue);
            return state == ExecutorState.REPEAT;
        }

        @Override
        public void submit() {
            ExecutorState state = STATE.getAndUpdate(this, ExecutorState::enqueue);
            if (state == ExecutorState.IDLE) {
                ComputerThread.this.queue(this);
            }
        }

        @Override
        public TimeoutState timeoutState() {
            return this.timeout;
        }

        @Override
        public long getRemainingTime() {
            return this.timeout.getRemainingTime();
        }

        @Override
        public void setRemainingTime(long time) {
            this.timeout.setRemainingTime(time);
        }
    }

    private final class WorkerThread
    implements Runnable {
        final int index;
        final Thread owner;
        final AtomicBoolean running = new AtomicBoolean(true);
        final AtomicReference<ExecutorImpl> currentExecutor = new AtomicReference<Object>(null);
        AtomicLong lastReport = new AtomicLong(Long.MIN_VALUE);

        WorkerThread(int index) {
            this.index = index;
            this.owner = workerFactory.newThread(this);
        }

        @Override
        public void run() {
            try {
                this.runImpl();
            }
            finally {
                ComputerThread.this.workerFinished(this);
            }
        }

        private void runImpl() {
            while (this.running.get()) {
                ExecutorImpl executor;
                ComputerThread.this.computerLock.lock();
                try {
                    ComputerThread.this.idleWorkers.getAndIncrement();
                    while ((executor = ComputerThread.this.computerQueue.pollFirst()) == null) {
                        if (ComputerThread.this.state.get() >= 1) {
                            return;
                        }
                        ComputerThread.this.workerWakeup.awaitUninterruptibly();
                    }
                }
                finally {
                    ComputerThread.this.idleWorkers.getAndDecrement();
                    ComputerThread.this.computerLock.unlock();
                }
                if (!ExecutorImpl.STATE.compareAndSet(executor, ExecutorState.ON_QUEUE, ExecutorState.RUNNING)) {
                    assert (false) : "Running computer on the wrong thread";
                    LOG.error("Trying to run computer #{} on thread {}, but already running on another thread. This is a SERIOUS bug, please report with your debug.log.", (Object)executor.worker.getComputerID(), (Object)this.owner.getName());
                }
                if (ComputerThread.this.state.get() >= 1) {
                    executor.worker.unload();
                }
                executor.beforeWork();
                this.currentExecutor.set(executor);
                try {
                    executor.worker.work();
                }
                catch (Exception | LinkageError | VirtualMachineError e) {
                    LOG.error("Error running task on computer #" + executor.worker.getComputerID(), e);
                    executor.worker.abortWithError();
                }
                finally {
                    ExecutorImpl thisExecutor = this.currentExecutor.getAndSet(null);
                    if (thisExecutor == null) continue;
                    ComputerThread.this.afterWork(executor);
                }
            }
        }

        private void reportTimeout(ExecutorImpl executor, long time) {
            if (!LOG.isErrorEnabled(Logging.COMPUTER_ERROR)) {
                return;
            }
            long now = System.nanoTime();
            long then = this.lastReport.get();
            if (then != Long.MIN_VALUE && now - then - REPORT_DEBOUNCE <= 0L) {
                return;
            }
            if (!this.lastReport.compareAndSet(then, now)) {
                return;
            }
            Thread owner = Objects.requireNonNull(this.owner);
            StringBuilder builder = new StringBuilder().append("Terminating computer #").append(executor.worker.getComputerID()).append(" due to timeout (ran over by ").append((double)time * -1.0E-9).append(" seconds). This is NOT a bug, but may mean a computer is misbehaving.\n").append("Thread ").append(owner.getName()).append(" is currently ").append((Object)owner.getState()).append('\n');
            Object blocking = LockSupport.getBlocker(owner);
            if (blocking != null) {
                builder.append("  on ").append(blocking).append('\n');
            }
            for (StackTraceElement element : owner.getStackTrace()) {
                builder.append("  at ").append(element).append('\n');
            }
            executor.worker.writeState(builder);
            LOG.warn(builder.toString());
        }
    }

    private final class Monitor
    implements Runnable {
        private Monitor() {
        }

        @Override
        public void run() {
            LOG.trace("Monitor starting.");
            try {
                this.runImpl();
            }
            finally {
                LOG.trace("Monitor shutting down. Current state is {}.", (Object)ComputerThread.this.state.get());
            }
        }

        private void runImpl() {
            while (ComputerThread.this.state.get() < 2) {
                ComputerThread.this.computerLock.lock();
                try {
                    ComputerThread.this.monitorWakeup.awaitNanos(ComputerThread.this.isBusy() ? ComputerThread.this.scaledPeriod() : MONITOR_WAKEUP);
                }
                catch (InterruptedException e) {
                    LOG.error("Monitor thread interrupted. Computers may behave very badly!", (Throwable)e);
                    break;
                }
                finally {
                    ComputerThread.this.computerLock.unlock();
                }
                this.checkRunners();
            }
        }

        private void checkRunners() {
            for (WorkerThread runner : ComputerThread.this.workersReadOnly()) {
                ExecutorImpl executor;
                if (runner == null || (executor = runner.currentExecutor.get()) == null) continue;
                executor.timeout.refresh();
                long remainingTime = executor.timeout.getRemainingTime();
                long afterHardAbort = -remainingTime - TimeoutState.ABORT_TIMEOUT;
                if (afterHardAbort < 0L) continue;
                executor.timeout.hardAbort();
                executor.worker.abortWithTimeout();
                if (afterHardAbort >= TimeoutState.ABORT_TIMEOUT * 2L) {
                    runner.reportTimeout(executor, remainingTime);
                    runner.owner.interrupt();
                    ComputerThread.this.workerFinished(runner);
                    continue;
                }
                if (afterHardAbort < TimeoutState.ABORT_TIMEOUT) continue;
                runner.reportTimeout(executor, remainingTime);
                runner.owner.interrupt();
            }
        }
    }

    private final class TimeoutImpl
    extends ManagedTimeoutState {
        private TimeoutImpl() {
        }

        @Override
        protected boolean shouldPause() {
            return ComputerThread.this.hasPendingWork();
        }
    }

    static final class ExecutorState
    extends Enum<ExecutorState> {
        public static final /* enum */ ExecutorState IDLE = new ExecutorState();
        public static final /* enum */ ExecutorState ON_QUEUE = new ExecutorState();
        public static final /* enum */ ExecutorState RUNNING = new ExecutorState();
        public static final /* enum */ ExecutorState REPEAT = new ExecutorState();
        private static final /* synthetic */ ExecutorState[] $VALUES;

        public static ExecutorState[] values() {
            return (ExecutorState[])$VALUES.clone();
        }

        public static ExecutorState valueOf(String name) {
            return Enum.valueOf(ExecutorState.class, name);
        }

        ExecutorState enqueue() {
            return switch (this.ordinal()) {
                default -> throw new MatchException(null, null);
                case 0, 1 -> ON_QUEUE;
                case 2, 3 -> REPEAT;
            };
        }

        ExecutorState requeue() {
            return switch (this.ordinal()) {
                default -> throw new MatchException(null, null);
                case 0, 1 -> {
                    if (!$assertionsDisabled) {
                        throw new AssertionError((Object)"Impossible state after executing");
                    }
                    LOG.error("Impossible state - calling requeue with {}.", (Object)this);
                    yield ON_QUEUE;
                }
                case 2 -> IDLE;
                case 3 -> ON_QUEUE;
            };
        }

        private static /* synthetic */ ExecutorState[] $values() {
            return new ExecutorState[]{IDLE, ON_QUEUE, RUNNING, REPEAT};
        }

        static {
            $VALUES = ExecutorState.$values();
        }
    }
}

